<?php

namespace App\Controller;

use App\Entity\Client;
use App\Repository\ClientRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class ClientController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;

    public function __construct(EntityManagerInterface $em, ValidatorInterface $validator, SerializerInterface $serializer)
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
    }

    # Render template
    #[Route('/clientes', name: 'clients.index', methods:['GET'])]  
    public function index(): Response
    {
        return $this->render('client/index.html.twig');
    }

    # GET /api/clients
    #[Route('/api/clients', name: 'api.clients.list', methods: ['GET'])]
    public function list(Request $request, ClientRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('razon_social');
        $type_documents = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $type_documents,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['client:read']]
        );
    }

    # GET /api/clients/:id
    #[Route('/api/clients/{id}', name: 'api.clients.show', methods: ['GET'])]
    public function show(Client $client): JsonResponse
    {
        return $this->json(
            data: $client,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['client:read']]
        );
    }

    # POST /api/clients
    #[Route('/api/clients', name: 'api.clients.create', methods: ['POST'])] 
    public function create(Request $request): JsonResponse
    {
        $data = $request->getContent();
        $client = $this->serializer->deserialize($data, Client::class, 'json');
        $this->em->persist($client);
        $this->em->flush();

        return $this->json(
            data: $client,
            status: Response::HTTP_CREATED,
            context: [AbstractNormalizer::GROUPS => ['client:read']]
        );
    }

    # PUT /api/clients/:id
    #[Route('/api/clients/{id}', name: 'api.clients.update', methods: ['PUT'])]
    public function update(Request $request, Client $client): JsonResponse
    {
        if (!$client) return $this->json([]); # por si no existe el usuario

        $data = $request->getContent();
        $this->serializer->deserialize($data, Client::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $client]);
        $this->em->flush();

        return $this->json(
            data: $client,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['client:read']]
        );
    }
}
