<?php

namespace App\Controller;

use App\Entity\Package;
use App\Repository\PackageRepository;
use App\Service\PdfGenerator;
use Doctrine\ORM\EntityManagerInterface;
use Dompdf\Dompdf;
use Dompdf\Options;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

final class PackageController extends AbstractController
{
    private $em;
    private $validator;
    private $serializer;

    public function __construct(EntityManagerInterface $em, ValidatorInterface $validator, SerializerInterface $serializer)
    {
        $this->em = $em;
        $this->validator = $validator;
        $this->serializer = $serializer;
    }

    # Render template package - paquetes
    #[Route('/paquetes', name: 'packages.index', methods:['GET'])]   
    public function index(): Response
    {
        return $this->render('package/index.html.twig');
    }

    # Render template create package - paquetes form
    #[Route('/paquetes/guardar/{id}', name: 'packages.save', methods:['GET'])]   
    public function save(int $id): Response
    {
        return $this->render('package/form.html.twig', [
            'title' => ($id === 0) ? 'Agregar paquete' : 'Editar paquete'
        ]);
    }

    #[Route('/paquetes/archivo/{id}', name: 'packages.file', methods:['GET'])]
    public function files(Package $package, PdfGenerator $pdfGenerator): Response
    {
        $data = array();
        $data['package'] = $package;
        // return $this->render('package/pdf.html.twig', $data);

        return new Response (
            $pdfGenerator->generatePdf('package/pdf.html.twig', $data),
            Response::HTTP_OK,
            ['Content-Type' => 'application/pdf']
        );
    }



    # GET /api/packages
    #[Route('/api/packages', name: 'api.packages.list', methods: ['GET'])]
    public function list(Request $request, PackageRepository $repo): JsonResponse
    {
        $nombre = $request->query->get('nombre');
        $packages = ($nombre) ? $repo->findByNombre($nombre) : $repo->findAll();

        return $this->json(
            data: $packages,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['package:read']] 
        );
    }

    # GET /api/packages/:id
    #[Route('/api/packages/{id}', name: 'api.packages.show', methods: ['GET'])]
    public function show(Package $package): JsonResponse
    {
        return $this->json(
            data: $package,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['package:read']]
        );
    }

    # POST /api/packages
    #[Route('/api/packages', name: 'api.packages.create', methods: ['POST'])]   
    public function create(Request $request, PackageRepository $repo): JsonResponse
    {
        $this->em->getConnection()->beginTransaction();

        try {
            $user = $this->getUser();
            $data = $request->getContent();
            $package = $this->serializer->deserialize($data, Package::class, 'json');
            $package->setUsuario($user);
            $package->setNumero($repo->getNextPackageNumber());
            
            $this->em->persist($package);
            $this->em->flush();
            $this->em->getConnection()->commit();

            return $this->json(
                data: $package,
                status: Response::HTTP_CREATED,
                context: [AbstractNormalizer::GROUPS => ['package:read']]
            );
        } catch (\Exception $e) {
            $this->em->getConnection()->rollBack();
            return $this->json($e->getMessage(), Response::HTTP_BAD_REQUEST);
        }
    }

    /*
    # PUT /api/countries/:id
    #[Route('/api/countries/{id}', name: 'api.countries.update', methods: ['PUT'])]
    public function update(Request $request, Country $country): JsonResponse
    {
        if (!$country) return $this->json([]); # por si no existe

        $data = $request->getContent();
        $this->serializer->deserialize($data, Country::class, 'json', [AbstractNormalizer::OBJECT_TO_POPULATE => $country]);
        $this->em->flush();

        return $this->json(
            data: $country,
            status: Response::HTTP_OK,
            context: [AbstractNormalizer::GROUPS => ['country:read']]
        );
    }
        */
}
